/*
    TiFileProp - command line tool to set TI68k variable file properties

    Copyright (C) 2008 Stefan Heule

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/


#include <stdio.h>
#include <stdlib.h>
#include <string.h>


#define TIFILEPROP_VERSION "1.0.1"
#define TIFILEPROP_RELEASE_DATE "2008-05-21"


const char* file_ext[] = {
"89a", "89g", "89d", "89c", "89e", "89f", "89g", "89i", "89k", "89l", "89m", "89p", "89s", "89t", "89u", "89x", "89y", "89z",
"9xa", "9xg", "9xd", "9xc", "9xe", "9xf", "9xg", "9xi", "9xk", "9xl", "9xm", "9xp", "9xs", "9xt", "9xu", "9xx", "9xy", "9xz",
"v2a", "v2g", "v2d", "v2c", "v2e", "v2f", "v2g", "v2i", "v2k", "v2l", "v2m", "v2p", "v2s", "v2t", "v2u", "v2x", "v2y", "v2z",
};

// -----------------------------------------------------------------
// prints the title
// -----------------------------------------------------------------
void PrintTitle(void)
{
    fprintf(stdout, "TiFileProp "TIFILEPROP_VERSION" - command line tool to set TI68k variable file properties\n");
    fprintf(stdout, "Copyright (C) 2008 Stefan Heule, "TIFILEPROP_RELEASE_DATE"\n");
}

// -----------------------------------------------------------------
// outputs usage information of this tool
// -----------------------------------------------------------------
void PrintUsage() {
    PrintTitle();
    fprintf(stdout, "\nUsage: tifileprop [flags] <infile>\n\n\n"\
                    "       -d <dest>   ... set the destination according <dest>:\n"\
                    "                        r .. RAM\n"\
                    "                        l .. RAM, locked\n"\
                    "                        a .. archive\n"\
                    "       -fo <name>  ... on-calc foldername (up to 8 characters)\n"\
                    "                       NOTE: use '-fo -' to send the file to the\n"\
                    "                             current folder of the receiving unit\n"\
                    "       -fi <name>  ... on-calc filename (up to 8 characters)\n"\
                    "       -c <str>    ... file comment (up to 40 characters)\n"\
                    "       -ti <calc>  ... set target device according <calc>:\n"\
                    "                        89 .. TI-89 (Titanium)\n"\
                    "                        9x .. TI-92 Plus\n"\
                    "                        v2 .. Voyage 200\n"\
                    "       -n_extcheck ... skip file extention check\n"\
                    "       -q          ... don't output standard messages\n\n"\
                    "       <infile>    ... input file\n\n");
}


// -----------------------------------------------------------------
// Return the filename extension or NULL
// -----------------------------------------------------------------
static const char *file_extension(const char *filename)
{
    int i;
    const char *p;

    for(i=strlen(filename); i > 0; i--)
    {
        if(filename[i] == '.') break;
    }
    p = filename+i+1;

    return p;
}


// -----------------------------------------------------------------
// our main function
// -----------------------------------------------------------------
int main(int argc, char **argv)
{
    FILE *ifp;
    char *infile = NULL;
    int dest = 0;
    char *foldername = NULL;
    char *filename = NULL;
    char *comment = NULL;
    char *calc =  NULL;
    int quiet = 0;
    int n_extcheck = 0;
    int i;
    int num_actions = 0;
    char tmp_name[41];
    int ext_ok = 0;

    // check for too less arguments
    if (argc < 2) {
        PrintUsage();
        return EXIT_FAILURE;
    }

    // parse arguments
    for (i = 1; i < argc; i++) {
        if (!strcmp(argv[i], "-q")) quiet = 1;
        else if (!strcmp(argv[i], "-n_extcheck")) n_extcheck = 1;
        else if (!strcmp(argv[i], "-d")) {
            if (i == argc-1) {
                PrintTitle();
                fprintf(stderr,"\nERROR: Invalid command line arguments, missing <dest> for '-d <dest>'!\n");
                return EXIT_FAILURE;
            }
            else {
                dest = *argv[i+1];
                if (dest != 'r' && dest != 'l' && dest != 'a') {
                    PrintTitle();
                    fprintf(stderr,"\nERROR: Invalid command line arguments, invalid <dest> for '-d <dest>'!\n");
                    return EXIT_FAILURE;
                }
                i++;
            }
        }
        else if (!strcmp(argv[i], "-fo")) {
            if (i == argc-1) {
                PrintTitle();
                fprintf(stderr,"\nERROR: Invalid command line arguments, missing <name> for '-fo <name>'!\n");
                return EXIT_FAILURE;
            }
            else {
                foldername = argv[i+1];
                i++;
            }
        }
        else if (!strcmp(argv[i], "-fi")) {
            if (i == argc-1) {
                PrintTitle();
                fprintf(stderr,"\nERROR: Invalid command line arguments, missing <name> for '-fi <name>'!\n");
                return EXIT_FAILURE;
            }
            else {
                filename = argv[i+1];
                i++;
            }
        }
        else if (!strcmp(argv[i], "-c")) {
            if (i == argc-1) {
                PrintTitle();
                fprintf(stderr,"\nERROR: Invalid command line arguments, missing <str> for '-c <str>'!\n");
                return EXIT_FAILURE;
            }
            else {
                comment = argv[i+1];
                i++;
            }
        }
        else if (!strcmp(argv[i], "-ti")) {
            if (i == argc-1) {
                PrintTitle();
                fprintf(stderr,"\nERROR: Invalid command line arguments, missing <calc> for '-ti <calc>'!\n");
                return EXIT_FAILURE;
            }
            else {
                calc = argv[i+1];
                if (strcmp(calc, "89") && strcmp(calc, "9x") && strcmp(calc, "v2")) {
                    PrintTitle();
                    fprintf(stderr,"\nERROR: Invalid command line arguments, invalid <calc> for '-ti <calc>'!\n");
                    return EXIT_FAILURE;
                }
                i++;
            }
        }
        else if (!infile) infile = argv[i];
        else {
            PrintTitle();
            fprintf(stderr,"\nERROR: Invalid command line argument: '%s'\n", argv[i]);
            return EXIT_FAILURE;
        }
    }

    // print the title
    if (!quiet) PrintTitle();

    // check file extension
    if (!n_extcheck) {
        const char* ext = file_extension(infile);
        for (i = 0; i < sizeof(file_ext)/sizeof(*file_ext); i++)
            if (!strcmp(ext, file_ext[i]))
                break;

        if (i == sizeof(file_ext)/sizeof(*file_ext)) {
            fprintf(stderr, "\nERROR: Wrong file extention!\n");
            return EXIT_FAILURE;
        }

        ext_ok = 1;
    }
    else {
        fprintf(stdout, "\nWARNING: File extention is not checked!");
    }

    // open inputfile
    if (!(ifp = fopen(infile, "r+b"))) {
        fprintf(stderr, "\nERROR: Cannot open inputfile %s!\n", infile);
        return EXIT_FAILURE;
    }

    // set the destination
    if (dest) {
        int val;
        if (dest == 'r') {
            val = 0;
            if (!quiet) fprintf(stdout, "\nTask %02i: Destination set to 'RAM'.", num_actions+1);
        }
        else if (dest == 'l') {
            val = 1;
            if (!quiet) fprintf(stdout, "\nTask %02i: Destination set to 'RAM, locked'.", num_actions+1);
        }
        else {
            val = 2;
            if (!quiet) fprintf(stdout, "\nTask %02i: Destination set to 'Archive'.", num_actions+1);
        }

        // write value
        fseek(ifp, 0x49, SEEK_SET);
        fputc(val, ifp);

        num_actions++;
    }

    // set the foldername
    if (foldername) {
        if (!strcmp(foldername, "-")) {
            fseek(ifp, 0x0a, SEEK_SET);
            fputc(0, ifp);

            if (!quiet) fprintf(stdout, "\nTask %02i: Send file to current folder of receiving unit.", num_actions+1);
        }
        else {
            strncpy(tmp_name, foldername, 8);
            tmp_name[8] = 0;
            if (strlen(foldername) > 8) fprintf(stdout, "\nWARNING: <foldername> exceeds 8 bytes. Truncated to fit!");

            if (!quiet) fprintf(stdout, "\nTask %02i: On-calc folder set to '%s'.", num_actions+1, tmp_name);

            // write foldername
            fseek(ifp, 0x0a, SEEK_SET);
            fwrite(tmp_name, 1, 8, ifp);
        }

        num_actions++;
    }

    // set the filename
    if (filename) {
        strncpy(tmp_name, filename, 8);
        tmp_name[8] = 0;
        if (strlen(filename) > 8) fprintf(stdout, "\nWARNING: <filename> exceeds 8 bytes. Truncated to fit!");

        if (!quiet) fprintf(stdout, "\nTask %02i: On-calc filename set to '%s'.", num_actions+1, tmp_name);

        // write filename
        fseek(ifp, 0x40, SEEK_SET);
        fwrite(tmp_name, 1, 8, ifp);

        num_actions++;
    }

    // set the comment
    if (comment) {
        strncpy(tmp_name, comment, 40);
        tmp_name[40] = 0;
        if (strlen(comment) > 40) fprintf(stdout, "\nWARNING: <comment> exceeds 40 bytes. Truncated to fit!");

        if (!quiet) fprintf(stdout, "\nTask %02i: File comment set.", num_actions+1);

        // write comment
        fseek(ifp, 0x12, SEEK_SET);
        fwrite(tmp_name, 1, 40, ifp);

        num_actions++;
    }

    // set target device
    if (calc) {
        fseek(ifp, 0x00, SEEK_SET);
        if (!strcmp(calc, "89")) {
            fputs("**TI89**", ifp);
        }
        else {
            fputs("**TI92P*", ifp);
        }

        fclose(ifp);
        ifp = NULL;

        // add new extension
        if (!n_extcheck && ext_ok) {
            char* tmp = malloc(strlen(infile)+1);
            if (!tmp) return EXIT_FAILURE;
            strcpy(tmp, infile);
            memcpy(tmp+strlen(infile)-3, calc, 2);
            rename(infile, tmp);
            free(tmp);
        }
        else {
            fprintf(stdout, "\nWARNING: Target device changed, but fileextension is left unmodified!");
        }

        if (!quiet) fprintf(stdout, "\nTask %02i: Target device set to '%s'", num_actions+1, calc);

        num_actions++;
    }

    // no action :(
    if (!num_actions) {
        fprintf(stderr, "\nWARNING: Nothing to do, use command line flags!");
    }

    // close the file
    if (ifp) fclose(ifp);

    if (!quiet) fprintf(stdout, "\n");

    return EXIT_SUCCESS;
}
















